# cli
# Copyright 2010-2011 Karl A. Knechtel.
#
# Utility functions for text wrapping and display.
#
# Licensed under the Generic Non-Commercial Copyleft Software License,
# Version 1.1 (hereafter "Licence"). You may not use this file except
# in the ways outlined in the Licence, which you should have received
# along with this file.
#
# Unless required by applicable law or agreed to in writing, software 
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied. See the License for the specific language governing
# permissions and limitations under the License.


def _wrap_line(line, max_length = 79, full_width = False):
	"""Wrap the line to a width of at most 'max_length' characters.
	If 'full_width' is set, newlines will be suppressed after
	lines that are exactly 'max_length' characters long, so that
	they won't appear double-spaced in a terminal window."""

	result = ''
	linelength = 0
	for word in line.split():
		required = len(word) + (0 if linelength == 0 else 1)
		if required + linelength > max_length:
			result += '\n' + word
			linelength = len(word)
		else:
			if linelength != 0: result += ' '
			result += word
			linelength += required
			if linelength == max_length:
				result += '' if full_width else '\n'
				linelength = 0
	return result.strip() # In case a newline was placed after the last word.


def wrap(text):
	return '\n\n'.join(_wrap_line(l) for l in text.split('\n\n') if l)


def display(text, suffix = '\n'):
	from sys import stdout as out
	out.write((wrap(text) + suffix).encode(out.encoding, 'replace'))
